<?php

namespace Tests\Feature;

use App\Domain\API\ApiCredentialService;
use App\Models\User;
use App\Support\Security\ApiSignatureService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;
use Tests\TestCase;

class WalletFlowTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_can_send_transfer_and_idempotency_prevents_duplicates(): void
    {
        $sender = User::factory()->create(['email_verified_at' => now()]);
        $receiver = User::factory()->create(['email_verified_at' => now()]);
        $sender->wallet->update(['available_balance' => 10000]);

        $payload = [
            'receiver_public_id' => $receiver->public_id,
            'amount' => 1500,
            'idempotency_key' => (string) Str::uuid(),
            'confirm' => '1',
        ];

        $response = $this->actingAs($sender)->post('/wallet/send', $payload);
        $response->assertRedirect(route('wallet.dashboard'));

        $this->assertDatabaseHas('transfers', [
            'sender_user_id' => $sender->id,
            'receiver_user_id' => $receiver->id,
            'amount' => 1500,
        ]);

        $this->actingAs($sender)->post('/wallet/send', $payload);

        $this->assertDatabaseCount('transfers', 1);
        $sender->refresh();
        $receiver->refresh();
        $this->assertSame(8500, $sender->wallet->available_balance);
        $this->assertSame(1500, $receiver->wallet->available_balance);
    }

    public function test_business_payment_link_single_use_flow(): void
    {
        $merchant = User::factory()->create(['email_verified_at' => now()]);
        $payer = User::factory()->create(['email_verified_at' => now()]);

        $merchant->businessProfile->update(['is_business_enabled' => true, 'business_name' => 'Shop']);
        $payer->wallet->update(['available_balance' => 10000]);

        $service = app(\App\Domain\Business\PaymentLinkService::class);

        $link = $service->createLink($merchant, [
            'title' => 'Invoice #1',
            'fixed_amount' => 2000,
            'allow_custom_amount' => false,
            'single_use' => true,
        ]);

        $service->payLink($link, $payer, 2000, (string) Str::uuid());

        $this->assertDatabaseHas('payment_links', [
            'id' => $link->id,
            'status' => 'paid',
        ]);

        $this->expectException(\Illuminate\Validation\ValidationException::class);
        $service->payLink($link->fresh(), $payer, 2000, (string) Str::uuid());
    }

    public function test_signed_api_replay_is_blocked(): void
    {
        $user = User::factory()->create(['email_verified_at' => now()]);
        $user->wallet->update(['available_balance' => 20000]);
        $receiver = User::factory()->create(['email_verified_at' => now()]);

        $creds = app(ApiCredentialService::class)->rotate($user);
        $key = $creds['key_id'];
        $secret = $creds['secret'];

        $payloadData = [
            'receiver_public_id' => $receiver->public_id,
            'amount' => 1000,
            'idempotency_key' => (string) Str::uuid(),
            'confirm' => true,
        ];
        $body = json_encode($payloadData, JSON_THROW_ON_ERROR);

        $timestamp = (string) now()->timestamp;
        $nonce = (string) Str::uuid();

        $signatureService = app(ApiSignatureService::class);
        $payload = $signatureService->buildPayload('POST', '/api/v1/transfers', $body, $timestamp, $nonce);
        $signature = $signatureService->sign($secret, $payload);

        $headers = [
            'X-API-KEY' => $key,
            'X-SIGNATURE' => $signature,
            'X-TIMESTAMP' => $timestamp,
            'X-NONCE' => $nonce,
        ];

        $first = $this->withHeaders($headers)->postJson('/api/v1/transfers', $payloadData);
        $first->assertOk();

        $second = $this->withHeaders($headers)->postJson('/api/v1/transfers', $payloadData);
        $second->assertStatus(401);
    }

    public function test_admin_user_update_creates_audit_record(): void
    {
        Role::findOrCreate('super_admin', 'web');

        $admin = User::factory()->create(['email_verified_at' => now()]);
        $admin->assignRole('super_admin');

        $target = User::factory()->create(['email_verified_at' => now()]);

        $response = $this->actingAs($admin)->patch(route('admin.users.update', $target), [
            'status' => 'blocked',
            'blocked_reason' => 'abuse',
            'daily_send_limit' => 1000,
            'monthly_send_limit' => 10000,
            'velocity_limit_per_minute' => 2,
            'can_send' => '1',
            'can_receive' => '1',
            'is_business_enabled' => '0',
            'can_api' => '1',
            'confirm' => '1',
        ]);

        $response->assertRedirect();

        $this->assertDatabaseHas('admin_actions_audits', [
            'admin_user_id' => $admin->id,
            'target_user_id' => $target->id,
            'action' => 'admin.user.update',
        ]);
    }
}
