<?php

namespace App\Http\Middleware;

use App\Models\ApiCredential;
use App\Models\User;
use App\Services\FeatureGateService;
use App\Support\Security\ApiSignatureService;
use App\Support\Security\NonceService;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class VerifyApiSignature
{
    public function __construct(
        private readonly ApiSignatureService $signatureService,
        private readonly NonceService $nonceService,
        private readonly FeatureGateService $featureGate,
    ) {
    }

    public function handle(Request $request, Closure $next): Response
    {
        $keyId = (string) $request->header('X-API-KEY', '');
        $signature = (string) $request->header('X-SIGNATURE', '');
        $timestamp = (string) $request->header('X-TIMESTAMP', '');
        $nonce = (string) $request->header('X-NONCE', '');

        if ($keyId === '' || $signature === '' || $timestamp === '' || $nonce === '') {
            return response()->json(['message' => 'Missing API signature headers.'], 401);
        }

        $credential = ApiCredential::query()->where('key_id', $keyId)->where('is_active', true)->first();
        if (! $credential) {
            return response()->json(['message' => 'Invalid API key.'], 401);
        }

        $payload = $this->signatureService->buildPayload(
            method: $request->method(),
            path: '/'.ltrim($request->path(), '/'),
            body: $request->getContent(),
            timestamp: $timestamp,
            nonce: $nonce,
        );

        if (! $this->signatureService->verify((string) $credential->secret_enc, $payload, $signature)) {
            return response()->json(['message' => 'Invalid signature.'], 401);
        }

        if (! ctype_digit($timestamp) || ! $this->nonceService->ensureFreshAndStore($keyId, $nonce, (int) $timestamp)) {
            return response()->json(['message' => 'Expired or replayed request.'], 401);
        }

        $user = User::query()->find($credential->user_id);
        if (! $user || $user->status !== 'active') {
            return response()->json(['message' => 'User not available.'], 401);
        }

        if (! $this->featureGate->isEnabled('api.access', $user)) {
            return response()->json(['message' => 'API access disabled.'], 403);
        }

        auth()->setUser($user);
        $credential->forceFill(['last_used_at' => now()])->save();

        return $next($request);
    }
}
