<?php

namespace App\Http\Controllers\Admin;

use App\Domain\Admin\AdminAuditService;
use App\Http\Controllers\Controller;
use App\Models\FeatureFlag;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;

class UserManagementController extends Controller
{
    public function __construct(private readonly AdminAuditService $audit)
    {
    }

    public function index(): View
    {
        return view('admin.users', [
            'users' => User::query()->with(['wallet', 'businessProfile', 'featureFlags'])->latest()->paginate(25),
        ]);
    }

    public function update(Request $request, User $user): RedirectResponse
    {
        $data = $request->validate([
            'status' => ['required', 'in:active,blocked,suspended'],
            'blocked_reason' => ['nullable', 'string', 'max:1000'],
            'business_name' => ['nullable', 'string', 'max:120'],
            'adjust_balance' => ['nullable', 'integer'],
            'daily_send_limit' => ['required', 'integer', 'min:1'],
            'monthly_send_limit' => ['required', 'integer', 'min:1'],
            'velocity_limit_per_minute' => ['required', 'integer', 'min:1', 'max:120'],
            'reason' => ['nullable', 'string', 'max:1000'],
            'confirm' => ['required', 'accepted'],
        ]);

        DB::transaction(function () use ($request, $user, $data): void {
            $canSend = $request->boolean('can_send');
            $canReceive = $request->boolean('can_receive');
            $isBusiness = $request->boolean('is_business_enabled');
            $canApi = $request->boolean('can_api');

            $user->update([
                'status' => $data['status'],
                'blocked_reason' => $data['blocked_reason'] ?? null,
            ]);

            $wallet = $user->wallet()->lockForUpdate()->firstOrFail();
            $wallet->can_send = $canSend;
            $wallet->can_receive = $canReceive;
            $wallet->daily_send_limit = (int) $data['daily_send_limit'];
            $wallet->monthly_send_limit = (int) $data['monthly_send_limit'];
            $wallet->velocity_limit_per_minute = (int) $data['velocity_limit_per_minute'];

            if (isset($data['adjust_balance']) && $data['adjust_balance'] !== 0) {
                $wallet->available_balance += (int) $data['adjust_balance'];
            }

            $wallet->save();

            $user->businessProfile()->update([
                'is_business_enabled' => $isBusiness,
                'business_name' => $data['business_name'] ?? null,
            ]);

            FeatureFlag::query()->updateOrCreate(
                ['scope' => 'user', 'user_id' => $user->id, 'feature_key' => 'api.access'],
                ['is_enabled' => $canApi, 'updated_by' => $request->user()->id],
            );

            $this->audit->log(
                admin: $request->user(),
                target: $user,
                action: 'admin.user.update',
                reason: $data['reason'] ?? null,
                meta: [
                    'status' => $data['status'],
                    'blocked_reason' => $data['blocked_reason'] ?? null,
                    'can_send' => $canSend,
                    'can_receive' => $canReceive,
                    'is_business_enabled' => $isBusiness,
                    'can_api' => $canApi,
                    'daily_send_limit' => (int) $data['daily_send_limit'],
                    'monthly_send_limit' => (int) $data['monthly_send_limit'],
                    'velocity_limit_per_minute' => (int) $data['velocity_limit_per_minute'],
                    'adjust_balance' => (int) ($data['adjust_balance'] ?? 0),
                ],
            );
        });

        return back()->with('status', 'User updated successfully.');
    }
}
