<?php

namespace App\Http\Controllers\Admin;

use App\Domain\Admin\AdminAuditService;
use App\Http\Controllers\Controller;
use App\Models\FeatureFlag;
use App\Models\SmtpSetting;
use App\Services\SystemSettingsService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class SettingsController extends Controller
{
    public function __construct(
        private readonly SystemSettingsService $settings,
        private readonly AdminAuditService $audit,
    ) {
    }

    public function index(): View
    {
        $featureKeys = [
            'wallet.transfer',
            'business.payment_links',
            'api.access',
        ];

        $globalFlags = FeatureFlag::query()
            ->where('scope', 'global')
            ->pluck('is_enabled', 'feature_key')
            ->toArray();

        return view('admin.settings', [
            'smtp' => SmtpSetting::query()->latest()->first(),
            'featureKeys' => $featureKeys,
            'globalFlags' => $globalFlags,
            'systemName' => $this->settings->get('system.name', config('app.name')),
            'copyright' => $this->settings->get('system.copyright', env('APP_COPYRIGHT', '')),
        ]);
    }

    public function updateSmtp(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'host' => ['required', 'string', 'max:255'],
            'port' => ['required', 'integer', 'min:1', 'max:65535'],
            'username' => ['nullable', 'string', 'max:255'],
            'password' => ['nullable', 'string', 'max:255'],
            'encryption' => ['nullable', 'in:tls,ssl'],
            'from_email' => ['required', 'email', 'max:255'],
            'from_name' => ['required', 'string', 'max:255'],
            'confirm' => ['required', 'accepted'],
        ]);

        $smtp = SmtpSetting::query()->latest()->first() ?? new SmtpSetting();
        $smtp->host = $data['host'];
        $smtp->port = (int) $data['port'];
        $smtp->username_enc = $data['username'] ?? null;
        $smtp->password_enc = ($data['password'] ?? '') !== '' ? $data['password'] : $smtp->password_enc;
        $smtp->encryption = $data['encryption'] ?? null;
        $smtp->from_email = $data['from_email'];
        $smtp->from_name = $data['from_name'];
        $smtp->updated_by = $request->user()->id;
        $smtp->save();

        $this->audit->log($request->user(), null, 'admin.settings.smtp.update', 'SMTP settings updated.');

        return back()->with('status', 'SMTP settings saved.');
    }

    public function updateFeatures(Request $request): RedirectResponse
    {
        $request->validate([
            'confirm' => ['required', 'accepted'],
        ]);

        $featureKeys = ['wallet.transfer', 'business.payment_links', 'api.access'];
        foreach ($featureKeys as $key) {
            FeatureFlag::query()->updateOrCreate(
                ['scope' => 'global', 'user_id' => null, 'feature_key' => $key],
                ['is_enabled' => $request->boolean("feature.{$key}"), 'updated_by' => $request->user()->id]
            );
        }

        $this->audit->log($request->user(), null, 'admin.settings.features.update', 'Global feature flags updated.');

        return back()->with('status', 'Feature flags saved.');
    }

    public function updateBranding(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'system_name' => ['required', 'string', 'max:100'],
            'copyright' => ['nullable', 'string', 'max:255'],
            'confirm' => ['required', 'accepted'],
        ]);

        $this->settings->set('system.name', $data['system_name']);
        $this->settings->set('system.copyright', $data['copyright'] ?? '');

        $this->audit->log($request->user(), null, 'admin.settings.branding.update', 'Branding updated.');

        return back()->with('status', 'Branding saved.');
    }
}
