<?php

namespace App\Domain\Wallet;

use App\Models\LedgerEntry;
use App\Models\Transfer;
use App\Models\User;
use App\Services\FeatureGateService;
use App\Services\TransferLimitService;
use App\Support\Security\TotpService;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class WalletTransferService
{
    public function __construct(
        private readonly TotpService $totpService,
        private readonly FeatureGateService $featureGate,
        private readonly TransferLimitService $transferLimitService,
    ) {
    }

    public function transfer(User $sender, User $receiver, int $amount, string $idempotencyKey, ?string $note = null, ?string $totpCode = null): Transfer
    {
        if (! $this->featureGate->isEnabled('wallet.transfer', $sender)) {
            throw ValidationException::withMessages(['feature' => 'Transfers are currently disabled.']);
        }

        if ($amount <= 0) {
            throw ValidationException::withMessages(['amount' => 'Amount must be positive.']);
        }

        if ($sender->id === $receiver->id) {
            throw ValidationException::withMessages(['receiver_public_id' => 'Cannot transfer to yourself.']);
        }

        if (! $this->totpService->verify($sender, $totpCode)) {
            throw ValidationException::withMessages(['totp_code' => 'Invalid 2FA code.']);
        }

        $this->transferLimitService->assertWithinLimits($sender, $amount);

        return DB::transaction(function () use ($sender, $receiver, $amount, $idempotencyKey, $note) {
            $existing = Transfer::query()->where('idempotency_key', $idempotencyKey)->first();
            if ($existing) {
                return $existing;
            }

            $senderWallet = $sender->wallet()->lockForUpdate()->firstOrFail();
            $receiverWallet = $receiver->wallet()->lockForUpdate()->firstOrFail();

            if (! $senderWallet->can_send || ! $receiverWallet->can_receive) {
                throw ValidationException::withMessages(['wallet' => 'Sending or receiving is disabled.']);
            }

            if ($sender->status !== 'active' || $receiver->status !== 'active') {
                throw ValidationException::withMessages(['wallet' => 'Account status does not allow transfer.']);
            }

            if ($senderWallet->available_balance < $amount) {
                throw ValidationException::withMessages(['amount' => 'Insufficient balance.']);
            }

            $senderWallet->available_balance -= $amount;
            $receiverWallet->available_balance += $amount;
            $senderWallet->save();
            $receiverWallet->save();

            $transfer = Transfer::query()->create([
                'sender_user_id' => $sender->id,
                'receiver_user_id' => $receiver->id,
                'amount' => $amount,
                'status' => 'completed',
                'idempotency_key' => $idempotencyKey,
                'note' => $note,
                'confirmed_at' => now(),
            ]);

            LedgerEntry::query()->create([
                'wallet_id' => $senderWallet->id,
                'direction' => 'debit',
                'amount' => $amount,
                'balance_after' => $senderWallet->available_balance,
                'reference_type' => 'transfer',
                'reference_id' => (string) $transfer->id,
            ]);

            LedgerEntry::query()->create([
                'wallet_id' => $receiverWallet->id,
                'direction' => 'credit',
                'amount' => $amount,
                'balance_after' => $receiverWallet->available_balance,
                'reference_type' => 'transfer',
                'reference_id' => (string) $transfer->id,
            ]);

            return $transfer;
        });
    }
}
